//==============================================================================
// Project:		Wealth Transfers and their Economic Effects
// File name:	Miscellaneous Bequest Recipient Data Analyses
// Objective: 	Calculating what share of the HILDA sample reported receiving an
//				inheritance after the death of their second parent.
//
// Created: 	25/05/2021
// Modified: 	15/11/2021
//==============================================================================

/////////////////////////////////////////////////////////////////////////////////////
/// Proportion Reporting Inheritances after the Death of their Last Living Parent ///
/////////////////////////////////////////////////////////////////////////////////////

**************************
*** Importing the Data ***
**************************

clear

cd "XXXX"	// set content directory here

use HILDA_restricted_combined_rescaled	// importing the data

xtset xwaveid wave	// declaring the dataset to be a panel

codebook psyodf psyodm	// checking the father and mother year of death variables

*************************
*** Cleaning the Data ***
*************************

replace psyodf = . if psyodf < 0	// replacing negative values of father's year of death
replace psyodm = . if psyodm < 0	// replacing negative values of mother's year of death

by xwaveid, sort: egen mother_yod = mean(psyodm)	// generating a mother's year of death variable
replace mother_yod = 0 if mother_yod == .			// replacing missing values of mother's year of death
replace mother_yod = round(mother_yod,1)	// rounding mother's year of death to the nearest integer

by xwaveid, sort: egen father_yod = mean(psyodf)	// generating a father's year of death variable
replace father_yod = 0 if father_yod == .			// replacing missing values of father's year of death
replace father_yod = round(father_yod,1)	// rounding father's year of death to the nearest integer

drop if mother_yod < 1999 & father_yod < 1999	// dropping individuals who had both parent's die before 1999
drop if mother_yod == 0 | father_yod == 0		// dropping if either mother or father's year of death is zero / missing

tab mother_yod	// tabulating mother's year of death
tab father_yod	// tabulating father's year of death

// generating a year of last parent's death variable
gen last_parent_death = .
replace last_parent_death = mother_yod if mother_yod > father_yod
replace last_parent_death = father_yod if father_yod >= mother_yod

gen death_wave = last_parent_death - 2000	// generating a variable equal to the wave that coincides with the year of last parent death

keep if wave <= death_wave + 4 & wave >= death_wave - 1	// keeping observations that were in the year prior or four years following the death of their last parent

save Expected_Recipients, replace	// saving the dataset

*************************************
*** Dropping Missing Observations ***
*************************************

* This section drops individuals who are not surveyed in the year of or two years subsequent to the death of their last living parent. This is done because the majority of inheritances are recevied during that window.

// identifying individuals who responded to the HILDA survey in the year when their last parent dies
keep if wave == death_wave
gen response = 1
collapse response, by(xwaveid)
save expected_recipients_temp1, replace

// identifying individuals who responded to the HILDA survey the year after their last living parent died
clear
use Expected_Recipients
keep if wave == death_wave + 1
gen response2 = 1
collapse response2, by(xwaveid)
save expected_recipients_temp2, replace

// identifying individuals who responded to the HILDA survey two years after their last living parent died
clear
use Expected_Recipients
keep if wave == death_wave + 2
gen response3 = 1
collapse response3, by(xwaveid)
save expected_recipients_temp3, replace

clear

use Expected_Recipients	// importing the data

// merging in the temporary datasets so as to drop respondents who are not in the sample in the year or two years following the death of their last living parent when most inheritances (>80%) are reported

// year of last parent death
merge m:1 xwaveid using expected_recipients_temp1
drop if _merge != 3
drop _merge

// year after last parent death
merge m:1 xwaveid using expected_recipients_temp2
drop if _merge != 3
drop _merge

// two years after last parent death
merge m:1 xwaveid using expected_recipients_temp3
drop if _merge != 3
drop _merge

save Expected_Recipients_Merged, replace

*****************************************
*** Proportion Reporting Inheritances ***
*****************************************

codebook oifinh	// checking the inheritance indicator variable

keep xwaveid wave sex yob yodeath mother_yod father_yod last_parent_death death_wave oifinh oifinha hgage	// keeping relevant variables

replace oifinha = 0 if oifinha < 0	// replacing negative values of inheritances as zero
replace oifinh = 0 if oifinh < 0	// replacing negative values of inheritances reported as zero

gen delay = wave - death_wave		// generating a variable equal to the difference between wave and the wave when the last parent died
tab delay if oifinh == 1			// tabulating the delay variable to see when people report inheritances

drop if yodeath <= last_parent_death + 1 & yodeath  > 0			// dropping individuals who die within a year of their last parent dying and thus might not have got a bequest yet
drop if last_parent_death >= 2020 | last_parent_death <= 2001	// dropping observations for 2000 and earlier and 2020 and later for which we think the bulk of bequests might be received outside the HILDA sample

by xwaveid, sort: egen num_bequests = sum(oifinh)	// generating a variable equal to the number of bequests individuals report

gen bequest = 0	// generating a bequest indicator variable
replace bequest = 1 if num_bequests >= 1	// replacing the bequest indicator variable equal to one if the respondent received one or more bequests

collapse bequest last_parent_death, by(xwaveid)

by last_parent_death, sort: tab bequest		// tabulating the number of bequests reported by year of last parent death

tab bequest	// tabulating the number of respondents who receive a bequest

gen num_obs = 1	// generating a variable that sums to the number of observations

collapse(sum) bequest num_obs, by(last_parent_death)	// collapsing the dataset to leave the number of bequest recipients by year of parent death

gen proportion = bequest / num_obs	// calculating the proportion of respondents who report a bequest following the death of their last living parent

save Expected_Recipients_Proportion, replace	// saving the dataset